<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class TransactionsController extends Controller
{
    public function index(): JsonResponse
    {
        // Eager load the payment, its related course, user, and user profiles
        $transactions = Transaction::with(['payment.course', 'payment.user.candidateProfile', 'payment.user.companyProfile'])->get();

        // Transform the data to include course title, user details, and profile details
        $transactions = $transactions->map(function ($transaction) {
            $user = $transaction->payment ? $transaction->payment->user : null;
            $profile = null;

            if ($user) {
                if ($user->User_type === 'candidate' && $user->candidateProfile) {
                    $profile = [
                        'type' => 'candidate',
                        'Profile_id' => $user->candidateProfile->Profile_id,
                        'First_name' => $user->candidateProfile->First_name,
                        'Last_name' => $user->candidateProfile->Last_name,
                        'Location' => $user->candidateProfile->Location,
                        'Resume_url' => $user->candidateProfile->Resume_url,
                        'Certifications' => $user->candidateProfile->Certifications,
                        'Experience' => $user->candidateProfile->Experience,
                        'Created_at' => $user->candidateProfile->Created_at,
                        'Updated_at' => $user->candidateProfile->Updated_at,
                    ];
                } elseif ($user->User_type === 'company' && $user->companyProfile) {
                    $profile = [
                        'type' => 'company',
                        'Company_id' => $user->companyProfile->Company_id,
                        'Company_name' => $user->companyProfile->Company_name,
                        'Location' => $user->companyProfile->Location,
                        'Description' => $user->companyProfile->Description,
                        'Logo_url' => $user->companyProfile->Logo_url,
                        'Plan_id' => $user->companyProfile->Plan_id,
                        'Subscription_id' => $user->companyProfile->Subscription_id,
                        'Created_at' => $user->companyProfile->Created_at,
                        'Updated_at' => $user->companyProfile->Updated_at,
                    ];
                }
            }

            return [
                'Transaction_id' => $transaction->Transaction_id,
                'Payment_id' => $transaction->Payment_id,
                'Transaction_type' => $transaction->Transaction_type,
                'Amount' => $transaction->Amount,
                'Currency' => $transaction->Currency,
                'Payment_method' => $transaction->Payment_method,
                'Status' => $transaction->Status,
                'Gateway_response' => $transaction->Gateway_response,
                'Processed_at' => $transaction->Processed_at,
                'created_at' => $transaction->created_at,
                'updated_at' => $transaction->updated_at,
                'payment' => $transaction->payment ? [
                    'Payment_id' => $transaction->payment->Payment_id,
                    'User_id' => $transaction->payment->User_id,
                    'Subscription_id' => $transaction->payment->Subscription_id,
                    'Coupon_id' => $transaction->payment->Coupon_id,
                    'Course_id' => $transaction->payment->Course_id,
                    'Course_name' => $transaction->payment->course ? $transaction->payment->course->Title : null,
                    'Amount' => $transaction->payment->Amount,
                    'Currency' => $transaction->payment->Currency,
                    'Payment_status' => $transaction->payment->Payment_status,
                    'Payment_gateway' => $transaction->payment->Payment_gateway,
                    'Transaction_id' => $transaction->payment->Transaction_id,
                    'Created_at' => $transaction->payment->Created_at,
                    'Updated_at' => $transaction->payment->Updated_at,
                    'user' => $user ? [
                        'User_id' => $user->User_id,
                        'Email' => $user->Email,
                        'User_type' => $user->User_type,
                        'Status' => $user->Status,
                        'Created_at' => $user->Created_at,
                        'Updated_at' => $user->Updated_at,
                        'profile' => $profile,
                    ] : null,
                ] : null,
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $transactions,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        // Eager load the payment, its related course, user, and user profiles
        $transaction = Transaction::with(['payment.course', 'payment.user.candidateProfile', 'payment.user.companyProfile'])->findOrFail($id);

        // Determine the profile based on User_type
        $user = $transaction->payment ? $transaction->payment->user : null;
        $profile = null;

        if ($user) {
            if ($user->User_type === 'candidate' && $user->candidateProfile) {
                $profile = [
                    'type' => 'candidate',
                    'Profile_id' => $user->candidateProfile->Profile_id,
                    'First_name' => $user->candidateProfile->First_name,
                    'Last_name' => $user->candidateProfile->Last_name,
                    'Location' => $user->candidateProfile->Location,
                    'Resume_url' => $user->candidateProfile->Resume_url,
                    'Certifications' => $user->candidateProfile->Certifications,
                    'Experience' => $user->candidateProfile->Experience,
                    'Created_at' => $user->candidateProfile->Created_at,
                    'Updated_at' => $user->candidateProfile->Updated_at,
                ];
            } elseif ($user->User_type === 'company' && $user->companyProfile) {
                $profile = [
                    'type' => 'company',
                    'Company_id' => $user->companyProfile->Company_id,
                    'Company_name' => $user->companyProfile->Company_name,
                    'Location' => $user->companyProfile->Location,
                    'Description' => $user->companyProfile->Description,
                    'Logo_url' => $user->companyProfile->Logo_url,
                    'Plan_id' => $user->companyProfile->Plan_id,
                    'Subscription_id' => $user->companyProfile->Subscription_id,
                    'Created_at' => $user->companyProfile->Created_at,
                    'Updated_at' => $user->companyProfile->Updated_at,
                ];
            }
        }

        // Transform the data to include course title, user details, and profile details
        $transactionData = [
            'Transaction_id' => $transaction->Transaction_id,
            'Payment_id' => $transaction->Payment_id,
            'Transaction_type' => $transaction->Transaction_type,
            'Amount' => $transaction->Amount,
            'Currency' => $transaction->Currency,
            'Payment_method' => $transaction->Payment_method,
            'Status' => $transaction->Status,
            'Gateway_response' => $transaction->Gateway_response,
            'Processed_at' => $transaction->Processed_at,
            'created_at' => $transaction->created_at,
            'updated_at' => $transaction->updated_at,
            'payment' => $transaction->payment ? [
                'Payment_id' => $transaction->payment->Payment_id,
                'User_id' => $transaction->payment->User_id,
                'Subscription_id' => $transaction->payment->Subscription_id,
                'Coupon_id' => $transaction->payment->Coupon_id,
                'Course_id' => $transaction->payment->Course_id,
                'Course_name' => $transaction->payment->course ? $transaction->payment->course->Title : null,
                'Amount' => $transaction->payment->Amount,
                'Currency' => $transaction->payment->Currency,
                'Payment_status' => $transaction->payment->Payment_status,
                'Payment_gateway' => $transaction->payment->Payment_gateway,
                'Transaction_id' => $transaction->payment->Transaction_id,
                'Created_at' => $transaction->payment->Created_at,
                'Updated_at' => $transaction->payment->Updated_at,
                'user' => $user ? [
                    'User_id' => $user->User_id,
                    'Email' => $user->Email,
                    'User_type' => $user->User_type,
                    'Status' => $user->Status,
                    'Created_at' => $user->Created_at,
                    'Updated_at' => $user->Updated_at,
                    'profile' => $profile,
                ] : null,
            ] : null,
        ];

        return response()->json([
            'status' => 'success',
            'data' => $transactionData,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Payment_id' => 'required|exists:payments,Payment_id',
            'Transaction_type' => 'required|in:Course Payment,Subscription Payment,Upgrade Plan,refund',
            'Amount' => 'required|numeric|min:0.01',
            'Currency' => 'required|string|size:3',
            'Payment_method' => 'required|in:razorpay,paypal,stripe',
            'Status' => 'required|in:pending,completed,failed',
            'Gateway_response' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $transaction = Transaction::create($request->only([
            'Payment_id',
            'Transaction_type',
            'Amount',
            'Currency',
            'Payment_method',
            'Status',
            'Gateway_response',
        ]) + ['Processed_at' => now()]);

        // Load the payment and course relationships for the response
        $transaction->load(['payment.course']);

        // Transform the data to include course title
        $transactionData = [
            'Transaction_id' => $transaction->Transaction_id,
            'Payment_id' => $transaction->Payment_id,
            'Transaction_type' => $transaction->Transaction_type,
            'Amount' => $transaction->Amount,
            'Currency' => $transaction->Currency,
            'Payment_method' => $transaction->Payment_method,
            'Status' => $transaction->Status,
            'Gateway_response' => $transaction->Gateway_response,
            'Processed_at' => $transaction->Processed_at,
            'created_at' => $transaction->created_at,
            'updated_at' => $transaction->updated_at,
            'payment' => $transaction->payment ? [
                'Payment_id' => $transaction->payment->Payment_id,
                'User_id' => $transaction->payment->User_id,
                'Subscription_id' => $transaction->payment->Subscription_id,
                'Coupon_id' => $transaction->payment->Coupon_id,
                'Course_id' => $transaction->payment->Course_id,
                'Course_name' => $transaction->payment->course ? $transaction->payment->course->Title : null,
                'Amount' => $transaction->payment->Amount,
                'Currency' => $transaction->payment->Currency,
                'Payment_status' => $transaction->payment->Payment_status,
                'Payment_gateway' => $transaction->payment->Payment_gateway,
                'Transaction_id' => $transaction->payment->Transaction_id,
                'Created_at' => $transaction->payment->Created_at,
                'Updated_at' => $transaction->payment->Updated_at,
            ] : null,
        ];

        return response()->json([
            'status' => 'success',
            'data' => $transactionData,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $transaction = Transaction::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Payment_id' => 'sometimes|exists:payments,Payment_id',
            'Transaction_type' => 'sometimes|in:Course Payment,Subscription Payment,Upgrade Plan,refund',
            'Amount' => 'sometimes|numeric|min:0.01',
            'Currency' => 'sometimes|string|size:3',
            'Payment_method' => 'sometimes|in:razorpay,paypal,stripe',
            'Status' => 'sometimes|in:pending,completed,failed',
            'Gateway_response' => 'sometimes|nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $transaction->update($request->only([
            'Payment_id',
            'Transaction_type',
            'Amount',
            'Currency',
            'Payment_method',
            'Status',
            'Gateway_response',
        ]));

        // Load the payment and course relationships for the response
        $transaction->load(['payment.course']);

        // Transform the data to include course title
        $transactionData = [
            'Transaction_id' => $transaction->Transaction_id,
            'Payment_id' => $transaction->Payment_id,
            'Transaction_type' => $transaction->Transaction_type,
            'Amount' => $transaction->Amount,
            'Currency' => $transaction->Currency,
            'Payment_method' => $transaction->Payment_method,
            'Status' => $transaction->Status,
            'Gateway_response' => $transaction->Gateway_response,
            'Processed_at' => $transaction->Processed_at,
            'created_at' => $transaction->created_at,
            'updated_at' => $transaction->updated_at,
            'payment' => $transaction->payment ? [
                'Payment_id' => $transaction->payment->Payment_id,
                'User_id' => $transaction->payment->User_id,
                'Subscription_id' => $transaction->payment->Subscription_id,
                'Coupon_id' => $transaction->payment->Coupon_id,
                'Course_id' => $transaction->payment->Course_id,
                'Course_name' => $transaction->payment->course ? $transaction->payment->course->Title : null,
                'Amount' => $transaction->payment->Amount,
                'Currency' => $transaction->payment->Currency,
                'Payment_status' => $transaction->payment->Payment_status,
                'Payment_gateway' => $transaction->payment->Payment_gateway,
                'Transaction_id' => $transaction->payment->Transaction_id,
                'Created_at' => $transaction->payment->Created_at,
                'Updated_at' => $transaction->payment->Updated_at,
            ] : null,
        ];

        return response()->json([
            'status' => 'success',
            'data' => $transactionData,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $transaction = Transaction::findOrFail($id);
        $transaction->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Transaction deleted successfully',
        ], 200);
    }
}
